"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUserSuggestions = exports.viewAllUsersClient = exports.viewAllUsers = exports.deleteUser = exports.updateUser = exports.addNewUser = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const users_1 = require("../../../../types/users");
const firestore_1 = require("firebase-admin/firestore");
const zod_1 = require("zod");
const auth_1 = require("firebase/auth");
const firebase_admin_2 = require("../../../../utils/firebase-admin");
const updateUserReferencesForName = (userId, updatedName) => __awaiter(void 0, void 0, void 0, function* () {
    // Update user's name in the `documents` collection
    const docsQuerySnapshot = yield firebase_admin_1.adminDb
        .collection("documents")
        .where("createdBy", "==", userId)
        .get();
    const batch = firebase_admin_1.adminDb.batch();
    docsQuerySnapshot.forEach((doc) => {
        batch.update(doc.ref, {
            createdByName: updatedName,
        });
    });
    // Commit the batch update
    yield batch.commit();
});
// const addNewUser = async (req: AuthenticatedRequest, res: Response) => {
//   try {
//     const currentUser = req.user;
//     const parsedBody: AddUser = addUserSchema.parse(req.body);
//     const { email, name, roleId, workspaceId, password } = parsedBody;
//     if (!["superadmin", "admin"].includes(currentUser?.role ?? "")) {
//       return res.status(403).json({
//         status: "error",
//         code: 403,
//         message: "Only superadmin and admin can perform this action.",
//       });
//     }
//     // Ensure that admins cannot add superadmins
//     if (currentUser!.role === "admin" && roleId === "superadmin") {
//       return res.status(403).json({
//         status: "error",
//         code: 403,
//         message: "Admins cannot add superadmins.",
//       });
//     }
//     // Check if the workspace exists and belongs to the current user's company
//     const workspaceSnapshot = await adminDb
//       .collection("workspaces")
//       .doc(workspaceId)
//       .get();
//     if (
//       !workspaceSnapshot.exists ||
//       workspaceSnapshot.data()?.companyId !== currentUser!.companyId
//     ) {
//       return res.status(400).json({
//         status: "error",
//         code: 400,
//         message: "Workspace does not belong to the current user's company.",
//       });
//     }
//     // Check if the role exists in the roles collection
//     const roleSnapshot = await adminDb.collection("roles").doc(roleId).get();
//     if (!roleSnapshot.exists) {
//       return res.status(400).json({
//         status: "error",
//         code: 400,
//         message: "Invalid role selected.",
//       });
//     }
//     // Check if the user already exists
//     const userSnapshot = await adminDb
//       .collection("users")
//       .where("email", "==", email)
//       .limit(1)
//       .get();
//     if (!userSnapshot.empty) {
//       return res.status(400).json({
//         status: "error",
//         code: 400,
//         message: "User already exists.",
//       });
//     }
//     // Encrypt the password
//     const encryptedPassword = await encryptPassword(password);
//     // Create a new user with default workspace
//     const newUserRef = adminDb.collection("users").doc();
//     const userId = newUserRef.id;
//     await newUserRef.set({
//       id: userId, // Adding the generated ID to the user document
//       email,
//       name,
//       password: encryptedPassword, // Encrypted password
//       companyId: currentUser!.companyId,
//       defaultWorkspace: workspaceId,
//       role: roleId,
//       createdAt: new Date().toISOString(),
//       createdBy: currentUser!.id,
//     });
//     return res.status(201).json({
//       status: "success",
//       code: 201,
//       message: "New user added successfully.",
//     });
//   } catch (error) {
//     if (error instanceof z.ZodError) {
//       const firstError = error.errors[0];
//       return res.status(400).json({
//         status: "error",
//         code: 400,
//         message: "Validation error",
//         error: firstError,
//       });
//     }
//     return res.status(500).json({
//       status: "error",
//       code: 500,
//       message: "Internal Server Error.",
//     });
//   }
// };
const authUser = (0, auth_1.getAuth)();
const addNewUser = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f;
    try {
        const currentUser = req.user;
        const parsedBody = users_1.addUserSchema.parse(req.body);
        const { email, name, roleId, workspaceId } = parsedBody;
        // Check if the current user is superadmin or admin
        if (!["superadmin", "admin"].includes((_a = currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) !== null && _a !== void 0 ? _a : "")) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmin and admin can perform this action.",
            });
        }
        // Ensure admins cannot add superadmins
        if (currentUser.role === "admin" && roleId === "superadmin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Admins cannot add superadmins.",
            });
        }
        // Check if the workspace exists and belongs to the current user's company
        const workspaceSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceSnapshot.exists ||
            ((_b = workspaceSnapshot.data()) === null || _b === void 0 ? void 0 : _b.companyId) !== currentUser.companyId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace does not belong to the current user's company.",
            });
        }
        // Check if the role exists in the roles collection
        const roleSnapshot = yield firebase_admin_1.adminDb.collection("roles").doc(roleId).get();
        if (!roleSnapshot.exists) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Invalid role selected.",
            });
        }
        // Check the company's user limit and current usage
        const companyRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(currentUser.companyId);
        const companySnapshot = yield companyRef.get();
        if (!companySnapshot.exists) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Company not found.",
            });
        }
        const companyData = companySnapshot.data();
        const userLimit = (_d = (_c = companyData === null || companyData === void 0 ? void 0 : companyData.plan) === null || _c === void 0 ? void 0 : _c.userLimit) !== null && _d !== void 0 ? _d : 0; // Get the user limit from the plan
        const userUsage = (_f = (_e = companyData === null || companyData === void 0 ? void 0 : companyData.usage) === null || _e === void 0 ? void 0 : _e.userUsage) !== null && _f !== void 0 ? _f : 0; // Get current user usage
        // Check if the user limit has been reached
        if (userUsage >= userLimit) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: `User limit reached. This company can have a maximum of ${userLimit} users.`,
            });
        }
        // Create a user in Firebase Auth without a password
        const newUser = yield firebase_admin_2.auth.createUser({
            email,
            displayName: name,
        });
        // Send password reset email to the new user
        yield (0, auth_1.sendPasswordResetEmail)(authUser, email);
        // Store user info in the company's user subcollection
        yield companyRef
            .collection("users")
            .doc(newUser.uid)
            .set({
            id: newUser.uid,
            email,
            name,
            companyId: currentUser.companyId,
            defaultWorkspace: workspaceId,
            roleId,
            createdAt: new Date().toISOString(),
            createdBy: currentUser.id,
            settings: {
                notifications: {
                    newDocumentNotification: true,
                    commentNotification: true,
                    approvalNotification: true,
                    taskNotification: true,
                },
            },
        });
        // Increment userUsage in the company's usage field
        yield companyRef.update({
            "usage.userUsage": firestore_1.FieldValue.increment(1),
        });
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "New user added successfully. Password reset email sent.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        else if (error.code === "auth/email-already-exists") {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "User with this email already exists.",
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.addNewUser = addNewUser;
const deleteUser = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const currentUser = req.user;
        const { userId } = req.params;
        // Fetch the user to be deleted from Firestore
        const companyRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(currentUser.companyId);
        const userRef = companyRef.collection("users").doc(userId);
        const userSnapshot = yield userRef.get();
        if (!userSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "User not found.",
            });
        }
        const userToDelete = userSnapshot.data();
        // Check if the user belongs to the same company
        if ((userToDelete === null || userToDelete === void 0 ? void 0 : userToDelete.companyId) !== currentUser.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You can only delete users in your own company.",
            });
        }
        // Only superadmins can delete superadmins
        if ((userToDelete === null || userToDelete === void 0 ? void 0 : userToDelete.role) === "superadmin" &&
            (currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) !== "superadmin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmin can delete another superadmin.",
            });
        }
        // Ensure admins cannot delete superadmins
        if ((currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) === "admin" && (userToDelete === null || userToDelete === void 0 ? void 0 : userToDelete.role) === "superadmin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Admins cannot delete superadmins.",
            });
        }
        // Delete the user in Firebase Auth and Firestore
        yield firebase_admin_2.auth.deleteUser(userId);
        yield userRef.delete();
        // Decrement userUsage in the company's usage field
        yield companyRef.update({
            "usage.userUsage": firestore_1.FieldValue.increment(-1),
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "User and associated data deleted successfully.",
        });
    }
    catch (error) {
        console.error("Error deleting user:", error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteUser = deleteUser;
const updateUser = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
    try {
        const currentUser = req.user;
        const _p = req.body, { roleId, email, password } = _p, updateData = __rest(_p, ["roleId", "email", "password"]); // Include name in updateData
        const { userId } = req.params;
        if (!userId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "User ID is required.",
            });
        }
        if (!roleId &&
            Object.keys(updateData).length === 0 &&
            !email &&
            !password) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "No data provided for update.",
            });
        }
        // Ensure the current user has the appropriate role
        if (!["superadmin", "admin"].includes((_a = currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) !== null && _a !== void 0 ? _a : "")) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmin and admin can perform this action.",
            });
        }
        // Ensure that admins cannot promote users to superadmin
        if ((currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) === "admin" && roleId === "superadmin") {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Admins cannot promote users to superadmin.",
            });
        }
        if (req.file && req.file.size > 5 * 1024 * 1024) {
            // 5 MB
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "File size should not exceed 5 MB.",
            });
        }
        // Fetch the user to be updated from Firestore
        const userRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(currentUser.companyId)
            .collection("users")
            .doc(userId);
        const userSnapshot = yield userRef.get();
        if (!userSnapshot.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "User not found.",
            });
        }
        const userToUpdate = userSnapshot.data();
        // Check if the user belongs to the same company as the current user
        if ((userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.companyId) !== currentUser.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You can only update users in your own company.",
            });
        }
        // 1. Update the Firebase Auth user if necessary
        const authUpdates = {};
        if (email && email !== (userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.email)) {
            authUpdates.email = email;
        }
        if (password)
            authUpdates.password = password;
        if (updateData.name)
            authUpdates.displayName = updateData.name; // Handle name in updateData
        if (Object.keys(authUpdates).length > 0) {
            yield firebase_admin_2.auth.updateUser(userId, authUpdates);
        }
        // Merge the update data with the existing user data
        const updatePayload = Object.assign(Object.assign(Object.assign({}, userToUpdate), updateData), { roleId: roleId || (userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.roleId), updatedAt: new Date().toISOString(), updatedBy: currentUser.id, settings: {
                notifications: {
                    newDocumentNotification: (_b = updateData.newDocumentNotification) !== null && _b !== void 0 ? _b : (_d = (_c = userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.settings) === null || _c === void 0 ? void 0 : _c.notifications) === null || _d === void 0 ? void 0 : _d.newDocumentNotification,
                    commentNotification: (_e = updateData.commentNotification) !== null && _e !== void 0 ? _e : (_g = (_f = userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.settings) === null || _f === void 0 ? void 0 : _f.notifications) === null || _g === void 0 ? void 0 : _g.commentNotification,
                    approvalNotification: (_h = updateData.approvalNotification) !== null && _h !== void 0 ? _h : (_k = (_j = userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.settings) === null || _j === void 0 ? void 0 : _j.notifications) === null || _k === void 0 ? void 0 : _k.approvalNotification,
                    taskNotification: (_l = updateData.taskNotification) !== null && _l !== void 0 ? _l : (_o = (_m = userToUpdate === null || userToUpdate === void 0 ? void 0 : userToUpdate.settings) === null || _m === void 0 ? void 0 : _m.notifications) === null || _o === void 0 ? void 0 : _o.taskNotification,
                },
            } });
        // Update Firestore email field if it was updated in Firebase Auth
        if (email) {
            updatePayload.email = email;
        }
        // Remove any undefined values from the update payload
        Object.keys(updatePayload).forEach((key) => updatePayload[key] === undefined && delete updatePayload[key]);
        // Update Firestore user document
        yield userRef.update(updatePayload);
        // If the name was updated, update the `createdByName` in documents
        if (updateData.name) {
            yield updateUserReferencesForName(userId, updateData.name);
        }
        // Fetch the updated user data to return
        const updatedUserSnapshot = yield userRef.get();
        const updatedUserData = updatedUserSnapshot.data();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "User updated successfully.",
            data: updatedUserData, // Return updated user data
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.updateUser = updateUser;
const sanitizeUserData = (user) => {
    const { password, refreshToken } = user, sanitizedUser = __rest(user, ["password", "refreshToken"]);
    return sanitizedUser;
};
const viewAllUsers = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const currentUser = req.user;
        // Ensure that the current user is either superadmin or admin
        if (!["superadmin", "admin"].includes((_a = currentUser === null || currentUser === void 0 ? void 0 : currentUser.role) !== null && _a !== void 0 ? _a : "")) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Only superadmin and admin can perform this action.",
            });
        }
        // Fetch users from Firestore under the current user's company
        const usersSnapshot = yield firebase_admin_1.adminDb
            .collection(`companies/${currentUser.companyId}/users`)
            .get();
        if (usersSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No users found for this company.",
            });
        }
        // Extract user IDs from Firestore documents
        const userIds = usersSnapshot.docs.map((doc) => doc.id);
        // Fetch the `photoURL` for each user from Firebase Authentication
        const userRecords = yield Promise.all(userIds.map((uid) => __awaiter(void 0, void 0, void 0, function* () {
            try {
                const userRecord = yield firebase_admin_2.auth.getUser(uid);
                return {
                    uid,
                    photoURL: userRecord.photoURL || null, // Add photoURL from Firebase Auth
                };
            }
            catch (error) {
                console.error(`Error fetching details for UID: ${uid}`, error.message);
                return { uid, photoURL: null }; // Return null if photoURL not found
            }
        })));
        // Create a map of photoURLs by UID for easy lookup
        const photoURLMap = new Map(userRecords.map((record) => [record.uid, record.photoURL]));
        // Map over the Firestore users and add the photoURL from Firebase Auth
        const users = usersSnapshot.docs.map((doc) => {
            const userData = doc.data();
            const uid = doc.id;
            return Object.assign(Object.assign({}, sanitizeUserData(userData)), { photoURL: photoURLMap.get(uid) });
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Users retrieved successfully.",
            data: users,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllUsers = viewAllUsers;
const viewAllUsersClient = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const currentUser = req.user;
        // Fetch users from Firestore under the current user's company
        const usersSnapshot = yield firebase_admin_1.adminDb
            .collection(`companies/${currentUser.companyId}/users`)
            .get();
        if (usersSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No users found for this company.",
            });
        }
        // Fetch user details from Firebase Authentication and Firestore
        const users = yield Promise.all(usersSnapshot.docs.map((doc) => __awaiter(void 0, void 0, void 0, function* () {
            const userFirestoreData = doc.data();
            try {
                const userRecord = yield firebase_admin_2.auth.getUser(doc.id);
                return {
                    id: userRecord.uid,
                    email: userRecord.email,
                    name: userRecord.displayName,
                    photoURL: userRecord.photoURL,
                    role: userFirestoreData.roleId, // Retrieve role from Firestore data
                };
            }
            catch (error) {
                console.error(`Failed to fetch auth details for user ${doc.id}:`, error);
                return null; // Return null if Firebase Auth details cannot be retrieved
            }
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Users retrieved successfully.",
            data: users,
        });
    }
    catch (error) {
        console.error("Error fetching users:", error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllUsersClient = viewAllUsersClient;
const getUserSuggestions = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        let { searchQuery } = req.query;
        const user = req.user;
        // Ensure the user is authenticated and part of a company
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Check if searchQuery exists and is a string
        if (typeof searchQuery !== "string" || !searchQuery.trim()) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Invalid or missing search query.",
            });
        }
        searchQuery = searchQuery.trim().toLowerCase();
        // Fetch users within the company based on the search query (email only)
        const userCollectionRef = firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users");
        const searchUpperBound = searchQuery + "\uf8ff";
        // Query by 'email' with "starts with" logic
        const usersSnapshot = yield userCollectionRef
            .where("email", ">=", searchQuery)
            .where("email", "<=", searchUpperBound)
            .limit(10)
            .get();
        // Check if users exist
        if (usersSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No users found.",
            });
        }
        // Extract user data
        const users = usersSnapshot.docs.map((doc) => {
            const userData = doc.data();
            return {
                id: doc.id,
                name: userData.name,
                email: userData.email,
            };
        });
        // Respond with the list of users
        return res.status(200).json({
            status: "success",
            code: 200,
            users,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.getUserSuggestions = getUserSuggestions;
