"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.viewManualDocumentReceiverApprovals = exports.viewManualDocumentApprovals = exports.deleteDocumentApproval = exports.editDocumentApproval = exports.viewDocumentReceiverApprovals = exports.viewDocumentApprovals = exports.submitDocumentApprovalDocuSeal = exports.sendManualDocumentApproval = exports.sendDocumentApprovalRequest = exports.generateDocuSealToken = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const notifications_1 = require("./notifications");
const firestore_1 = require("firebase/firestore");
const firebase_1 = require("../../../../utils/firebase");
const axios_1 = __importDefault(require("axios"));
const email_1 = require("../../../../utils/email");
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const crypto_1 = __importDefault(require("crypto"));
const generateDocumentKey = () => {
    return crypto_1.default.randomBytes(16).toString("hex"); // Example 16-byte random key
};
const cellExtensions = [
    "csv",
    "et",
    "ett",
    "fods",
    "ods",
    "ots",
    "sxc",
    "xls",
    "xlsb",
    "xlsm",
    "xlsx",
    "xlt",
    "xltm",
    "xltx",
    "xml",
];
const generateDocuSealToken = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    try {
        const { workspaceId, documentId, divisiFolderId } = req.params;
        const { userId } = req.body;
        const user = req.user;
        if (!workspaceId || !documentId || !divisiFolderId || !userId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID, Document ID, Folder ID, and Superior ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to send document approvals.",
            });
        }
        const [workspaceDoc, documentDoc] = yield Promise.all([
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "workspaces", workspaceId)),
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "documents", documentId)),
        ]);
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        if (!documentDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        const documentData = documentDoc.data();
        if (documentData.divisiFolderId !== divisiFolderId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Document does not belong to the specified folder.",
            });
        }
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const fileExtension = (documentData === null || documentData === void 0 ? void 0 : documentData.fileExtension) || "docx";
        const documentTitle = (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document";
        if (!filePath) {
            return res.status(500).json({
                status: "error",
                message: "Document storage path is missing in the database.",
            });
        }
        const { access } = documentData;
        const hasPermission = access.ownerId === user.id ||
            ((_a = access.allowedRoles) === null || _a === void 0 ? void 0 : _a.includes(user.role)) ||
            ((_b = access.invitedUsers) === null || _b === void 0 ? void 0 : _b.some((invitedUser) => invitedUser.id === user.id &&
                ["editor", "viewer"].includes(invitedUser.role)));
        if (!hasPermission) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to request approvals for this document.",
            });
        }
        const usersCollection = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        const [ownerDoc, requesterDoc] = yield Promise.all([
            (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, access.ownerId)),
            (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, user.id)),
        ]);
        if (!ownerDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document owner not found.",
            });
        }
        if (!requesterDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Requester not found.",
            });
        }
        const superiorDocs = yield Promise.all((Array.isArray(userId) ? userId : [userId]).map((id) => (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, id))));
        const invalidSuperiors = superiorDocs.filter((doc) => !doc.exists());
        if (invalidSuperiors.length > 0) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "One or more superior users not found.",
            });
        }
        const superiorData = superiorDocs.map((doc) => doc.data());
        const receiver = superiorData.map((superior) => ({
            external_id: superior.id,
            name: superior.name,
            email: superior.email,
        }));
        const documentRef = (0, firestore_1.doc)(firebase_1.db, "documents", documentId);
        let existingPdfSignedUrl = null;
        // If the file is already a PDF
        if (fileExtension === "pdf") {
            const existingPdfFilePath = filePath;
            const storagePdfFile = firebase_admin_1.bucket.file(existingPdfFilePath);
            [existingPdfSignedUrl] = yield storagePdfFile.getSignedUrl({
                action: "read",
                expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
            });
            const tokenDocuseal = jsonwebtoken_1.default.sign({
                user_email: process.env.DOCUSEAL_USER_EMAIL || "unknown",
                integration_email: user.email,
                external_id: documentId,
                name: `${documentTitle}`,
                document_urls: [existingPdfSignedUrl],
            }, process.env.DOCUSEAL_API_KEY || "", {});
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "Document is already a PDF. Token generated successfully.",
                data: {
                    tokenDocuseal,
                    receiver,
                    storagePath: existingPdfFilePath,
                },
            });
        }
        if (documentData === null || documentData === void 0 ? void 0 : documentData.documentPDF) {
            if (((_c = documentData === null || documentData === void 0 ? void 0 : documentData.documentPDF) === null || _c === void 0 ? void 0 : _c.documentBaseVersion) ===
                (documentData === null || documentData === void 0 ? void 0 : documentData.currentVersion)) {
                const existingPdfFilePath = documentData.documentPDF.storagePath;
                const storagePdfFile = firebase_admin_1.bucket.file(existingPdfFilePath);
                [existingPdfSignedUrl] = yield storagePdfFile.getSignedUrl({
                    action: "read",
                    expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
                });
                const tokenDocuseal = jsonwebtoken_1.default.sign({
                    user_email: process.env.DOCUSEAL_USER_EMAIL || "unknown",
                    integration_email: user.email,
                    external_id: documentId,
                    name: `${documentTitle}`,
                    document_urls: [existingPdfSignedUrl],
                }, process.env.DOCUSEAL_API_KEY || "", {});
                return res.status(200).json({
                    status: "success",
                    code: 200,
                    message: "Document is already a PDF. Token generated successfully.",
                    data: {
                        tokenDocuseal,
                        receiver,
                        storagePath: existingPdfFilePath,
                    },
                });
            }
        }
        // Otherwise, initiate a conversion
        const storageFile = firebase_admin_1.bucket.file(filePath);
        const [signedUrl] = yield storageFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        const key = generateDocumentKey();
        const payload = Object.assign({ async: false, filetype: fileExtension, key, outputtype: "pdf", title: documentTitle, url: existingPdfSignedUrl || signedUrl }, (cellExtensions.includes(fileExtension) && {
            spreadsheetLayout: {
                ignorePrintArea: true,
                orientation: "landscape",
                fitToWidth: 1,
                fitToHeight: 1,
                scale: 100,
                headings: false,
                gridLines: false,
                pageSize: {
                    width: "210mm",
                    height: "148mm",
                },
            },
        }));
        const token = jsonwebtoken_1.default.sign(payload, process.env.JWT_SECRET_KEY_ONLYOFFICE || "", {});
        const requestPayload = Object.assign(Object.assign({}, payload), { token });
        const documentServerUrl = process.env.ONLYOFFICE_DOCUMENT_SERVER_URL ||
            "http://localhost:80/converter";
        const conversionUrl = `${documentServerUrl}?shardkey=${key}`;
        const response = yield axios_1.default.post(conversionUrl, requestPayload, {
            headers: { "Content-Type": "application/json" },
        });
        const conversionResult = response.data;
        if (conversionResult.error || !conversionResult.endConvert) {
            return res.status(500).json({
                status: "error",
                message: `Conversion failed: ${conversionResult.error}`,
            });
        }
        const pdfUrl = conversionResult.fileUrl;
        // Fetch the PDF
        const pdfResponse = yield fetch(pdfUrl);
        if (!pdfResponse.ok) {
            return res.status(500).json({
                status: "error",
                message: "Failed to download converted PDF from ONLYOFFICE.",
            });
        }
        const pdfBuffer = yield pdfResponse.arrayBuffer();
        const fileSizeMB = (pdfBuffer.byteLength / (1024 * 1024)).toFixed(2); // Calculate file size in MB
        // Generate the new file path
        const baseFilePath = filePath.replace(/\/[^/]+$/, "");
        const pdfFilePath = `${baseFilePath}/pdf/${documentTitle}.pdf`;
        // Save the PDF to Firebase Storage
        const pdfStorageFile = firebase_admin_1.bucket.file(pdfFilePath);
        yield pdfStorageFile.save(Buffer.from(pdfBuffer), {
            resumable: false,
            contentType: "application/pdf",
        });
        // Fetch user details: superior, owner, requester
        yield (0, firestore_1.updateDoc)(documentRef, {
            documentPDF: {
                documentBaseVersion: documentData.currentVersion,
                storagePath: pdfFilePath,
                fileSizeMB,
                createdAt: new Date().toISOString(),
            },
        });
        if (!pdfFilePath) {
            return res.status(500).json({
                status: "error",
                message: "Document storage path is missing in the database.",
            });
        }
        const storagePdfFile = firebase_admin_1.bucket.file(pdfFilePath);
        const [pdfSignedUrl] = yield storagePdfFile.getSignedUrl({
            action: "read",
            expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
        });
        const tokenDocuseal = jsonwebtoken_1.default.sign({
            user_email: process.env.DOCUSEAL_USER_EMAIL || "unknown",
            integration_email: user.email,
            external_id: documentId,
            name: `${documentTitle} (Converted PDF)`,
            document_urls: [pdfSignedUrl],
        }, process.env.DOCUSEAL_API_KEY || "", {});
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval request has been sent and stored successfully.",
            data: {
                tokenDocuseal,
                receiver,
                storagePath: pdfFilePath,
            },
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.generateDocuSealToken = generateDocuSealToken;
const sendToDocuSealSubmission = (_a) => __awaiter(void 0, [_a], void 0, function* ({ templateId, submitters, }) {
    try {
        const payload = {
            template_id: templateId,
            send_email: false,
            submitters,
        };
        const response = yield axios_1.default.post("https://api.docuseal.com/submissions", payload, {
            headers: {
                "X-Auth-Token": process.env.DOCUSEAL_API_KEY,
                "Content-Type": "application/json",
            },
        });
        return response.data;
    }
    catch (error) {
        throw new Error("Failed to send document submission to DocuSeal.");
    }
});
const sendDocumentApprovalRequest = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { workspaceId, documentId, divisiFolderId } = req.params;
        const { userId, storagePath, templateId, submitters } = req.body;
        const user = req.user;
        // Validate request data
        if (!workspaceId || !documentId || !divisiFolderId || !userId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID, Document ID, Folder ID, and Superior ID(s) are required, and Superior ID must be an array.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to send document approvals.",
            });
        }
        // Fetch workspace and document
        const [workspaceDoc, documentDoc] = yield Promise.all([
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "workspaces", workspaceId)),
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "documents", documentId)),
        ]);
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        if (!documentDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        const documentData = documentDoc.data();
        // Validate folder ownership
        if (documentData.divisiFolderId !== divisiFolderId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Document does not belong to the specified folder.",
            });
        }
        const documentTitle = (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document";
        const { access } = documentData;
        const hasPermission = access.ownerId === user.id ||
            ((_a = access.allowedRoles) === null || _a === void 0 ? void 0 : _a.includes(user.role)) ||
            ((_b = access.invitedUsers) === null || _b === void 0 ? void 0 : _b.some((invitedUser) => invitedUser.id === user.id &&
                ["editor", "viewer"].includes(invitedUser.role)));
        if (!hasPermission) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to request approvals for this document.",
            });
        }
        // Fetch user details: superiors, owner, requester
        const usersCollection = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        const superiors = yield Promise.all((Array.isArray(userId) ? userId : [userId]).map((id) => (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, id))));
        const [ownerDoc, requesterDoc] = yield Promise.all([
            (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, access.ownerId)),
            (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, user.id)),
        ]);
        if (!ownerDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document owner not found.",
            });
        }
        if (!requesterDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Requester not found.",
            });
        }
        const ownerData = ownerDoc.data();
        const requesterData = requesterDoc.data();
        const superiorData = superiors
            .filter((doc) => doc.exists())
            .map((doc) => doc.data());
        if (superiorData.length === 0) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No valid superior users found.",
            });
        }
        const invitedUsers = superiorData.map((superior) => ({
            id: superior.id,
            name: superior.name,
            email: superior.email,
            photoURL: superior.photoURL || null,
            role: "editor",
        }));
        const documentApprovalRef = firebase_admin_1.adminDb.collection("documentApprovals").doc();
        const docuSealResponse = yield sendToDocuSealSubmission({
            templateId,
            submitters,
        });
        const newApproval = {
            id: documentApprovalRef.id,
            title: `${documentTitle} (Converted PDF)`,
            access: {
                allowedRoles: ["superadmin", "admin"],
                ownerId: user.id,
                ownerName: user.name,
                invitedUsers,
            },
            workspaceId,
            documentId,
            divisiFolderId,
            isManual: false,
            requesterId: user.id,
            receiver: userId.map((id, index) => ({
                id,
                slug: docuSealResponse[index].slug,
                isSigned: false,
            })),
            status: "Dalam Proses",
            requestedAt: new Date().toISOString(),
            fileExtension: "pdf",
            storagePath,
        };
        yield documentApprovalRef.set(newApproval);
        const emailLink = `${process.env.APP_URL}/document-approval`;
        const notifySuperiors = superiorData.filter((superior) => { var _a, _b; return ((_b = (_a = superior.settings) === null || _a === void 0 ? void 0 : _a.notifications) === null || _b === void 0 ? void 0 : _b.approvalNotification) === true; });
        const recipients = notifySuperiors.map((superior) => ({
            id: superior.id,
            name: superior.name,
            email: superior.email,
            photoURL: superior.photoURL || "",
        }));
        yield (0, notifications_1.sendNotificationDocumentApproval)(documentApprovalRef.id, documentData.id, documentData.title, requesterData.companyId, user.id, user.name, user.photoURL || undefined, recipients, documentData.workspaceId, false, "document approval submited", `You have been requested to review and approve the document titled "${documentData.title}".`, // Message
        "approvalNotification");
        yield Promise.all(notifySuperiors.map((superior) => (0, email_1.sendEmailApproval)({
            toEmail: superior.email,
            fromName: requesterData.name,
            recipientName: superior.name,
            link: `${emailLink}/e-signs`,
            message: `You have been requested to review and approve the document titled "${documentData.title}". \nThank you.`,
        })));
        // Notify requester
        yield (0, email_1.sendEmailApproval)({
            toEmail: requesterData.email,
            fromName: "System Notification",
            recipientName: requesterData.name,
            link: `${emailLink}/e-requests`,
            message: `Your request for document approval has been successfully sent for the document titled "${documentData.title}".\nThank you.`,
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval request has been sent and stored successfully.",
        });
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.sendDocumentApprovalRequest = sendDocumentApprovalRequest;
const submitDocumentApprovalDocuSeal = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { documentApprovalId } = req.params;
        const { submissionId } = req.body;
        const user = req.user;
        // Validate request
        if (!submissionId || !documentApprovalId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Submission ID and Document Approval ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to perform this action.",
            });
        }
        // Fetch submission data from DocuSeal API
        // Reference Firestore document
        const documentApprovalRef = (0, firestore_1.doc)(firebase_1.db, "documentApprovals", documentApprovalId);
        const documentApprovalSnap = yield (0, firestore_1.getDoc)(documentApprovalRef);
        if (!documentApprovalSnap.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document approval not found.",
            });
        }
        const documentApprovalData = documentApprovalSnap.data();
        const options = {
            method: "GET",
            url: `https://api.docuseal.com/submissions/${submissionId}`,
            headers: { "X-Auth-Token": process.env.DOCUSEAL_API_KEY },
        };
        const response = yield axios_1.default.request(options);
        const submissionData = response.data;
        if (submissionData.status === "completed") {
            const documentId = documentApprovalData.documentId;
            if (documentId) {
                const documentRef = (0, firestore_1.doc)(firebase_1.db, "documents", documentId);
                yield (0, firestore_1.updateDoc)(documentRef, {
                    status: "disetujui",
                    updatedAt: new Date().toISOString(),
                });
            }
        }
        let storagePath = "";
        let updatedTitle = documentApprovalData.title || "Document Approval";
        const requesterId = documentApprovalData.requesterId; // Adjust this field name based on your schema
        const requesterDoc = yield firebase_admin_1.adminDb
            .collection("companies")
            .doc(user.companyId)
            .collection("users")
            .doc(requesterId)
            .get();
        if (!requesterDoc.exists) {
            console.error(`Requester with ID ${requesterId} not found.`);
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Requester not found.",
            });
        }
        const requesterData = requesterDoc.data();
        for (const submitter of submissionData.submitters) {
            const receiverItem = documentApprovalData.receiver.find((receiver) => receiver.id === submitter.external_id);
            if (receiverItem) {
                const isCurrentlySigned = receiverItem.isSigned;
                if (submitter.status === "completed") {
                    receiverItem.isSigned = true;
                    const allSigned = documentApprovalData.receiver.every((receiver) => receiver.isSigned === true);
                    if (!isCurrentlySigned) {
                        // Update the title with the latest signer
                        if (allSigned) {
                            updatedTitle = `${documentApprovalData.title || "Document Approval"} - All Signed`;
                        }
                        else {
                            updatedTitle = `${documentApprovalData.title || "Document Approval"} - Signed by ${submitter.name || "Unknown Signer"}`;
                        }
                        // Get document info for this submitter
                        const documentInfo = (_a = submitter.documents) === null || _a === void 0 ? void 0 : _a[0];
                        if (!documentInfo)
                            continue;
                        const documentUrl = documentInfo.url;
                        const originalDocumentName = documentInfo.name || "SignedDocument";
                        const suffix = allSigned
                            ? "AllSigned"
                            : `Signed-${submitter.name || "UnknownSigner"}`;
                        const finalDocumentName = `${originalDocumentName}-${suffix}.pdf`;
                        const pdfResponse = yield fetch(documentUrl);
                        if (!pdfResponse.ok) {
                            return res.status(500).json({
                                status: "error",
                                message: "Failed to download signed PDF from DocuSeal.",
                            });
                        }
                        const pdfBuffer = yield pdfResponse.arrayBuffer();
                        // Generate a consistent storage path to overwrite the file
                        storagePath = `${user.companyId}/workspaces/${documentApprovalData.workspaceId}/approvals/${documentApprovalId}/${finalDocumentName}`;
                        const pdfStorageFile = firebase_admin_1.bucket.file(storagePath);
                        // Save (overwrite) the document to Firebase Storage
                        yield pdfStorageFile.save(Buffer.from(pdfBuffer), {
                            resumable: false,
                            contentType: "application/pdf",
                        });
                        console.log(`Document saved for submitter: ${submitter.external_id} at path: ${storagePath}`);
                    }
                    const notificationRecipients = [
                        {
                            id: requesterData.id,
                            name: requesterData.name,
                            email: requesterData.email,
                            photoURL: requesterData.photoURL || "",
                            isRead: false,
                        },
                    ];
                    const notificationMessage = allSigned
                        ? `The document "${documentApprovalData.title}" has been signed by all participants.`
                        : `The document "${documentApprovalData.title}" has been signed by ${submitter.name || "Unknown Signer"}.`;
                    // Send notification to all relevant recipients
                    yield (0, notifications_1.sendNotificationDocumentApproval)(documentApprovalId, documentApprovalData.documentId, documentApprovalData.title, user.companyId, user.id, user.name, user.photoURL || undefined, notificationRecipients, documentApprovalData.workspaceId, false, "document approval signed", notificationMessage, "approvalNotification");
                }
            }
        }
        yield (0, firestore_1.updateDoc)(documentApprovalRef, {
            receiver: documentApprovalData.receiver,
            updatedAt: new Date().toISOString(),
            key: generateDocumentKey(),
            storagePath: storagePath,
            title: updatedTitle, // Update the title in Firestore
        });
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval submission processed successfully for all submitters.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.submitDocumentApprovalDocuSeal = submitDocumentApprovalDocuSeal;
const viewDocumentApprovals = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.id || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Unauthorized access.",
            });
        }
        const documentApprovalsRef = (0, firestore_1.collection)(firebase_1.db, "documentApprovals");
        const querySnapshot = (0, firestore_1.query)(documentApprovalsRef, (0, firestore_1.and)((0, firestore_1.where)("workspaceId", "==", workspaceId), (0, firestore_1.where)("isManual", "==", false)));
        const approvalsSnapshot = yield (0, firestore_1.getDocs)(querySnapshot);
        if (approvalsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No document approvals found for this workspace.",
            });
        }
        const approvals = [];
        const usersRef = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        for (const approvalDoc of approvalsSnapshot.docs) {
            const data = approvalDoc.data();
            // Check if the current user is the requester
            const isRequester = data.requesterId === user.id;
            if (!isRequester) {
                // Skip if the user is not the requester
                continue;
            }
            // Fetch requester details
            const requesterSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, data.requesterId));
            const requesterData = requesterSnapshot.exists()
                ? requesterSnapshot.data()
                : null;
            // Fetch details for all receivers
            const receivers = yield Promise.all(data.receiver.map((receiver) => __awaiter(void 0, void 0, void 0, function* () {
                const receiverSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, receiver.id));
                return receiverSnapshot.exists()
                    ? {
                        id: receiver.id,
                        name: receiverSnapshot.data().name || null,
                        photoURL: receiverSnapshot.data().photoURL || null,
                        email: receiverSnapshot.data().email || null,
                        slug: receiver.slug,
                        isSigned: receiver.isSigned,
                    }
                    : null;
            })));
            approvals.push({
                documentData: data,
                requester: requesterData
                    ? {
                        id: data.requesterId,
                        name: requesterData.name || null,
                        photoURL: requesterData.photoURL || null,
                        email: requesterData.email || null,
                    }
                    : null,
                receiver: receivers.filter((receiver) => receiver !== null),
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            data: approvals,
        });
    }
    catch (error) {
        console.error(error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.viewDocumentApprovals = viewDocumentApprovals;
const viewDocumentReceiverApprovals = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.id || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Unauthorized access.",
            });
        }
        const documentApprovalsRef = (0, firestore_1.collection)(firebase_1.db, "documentApprovals");
        const querySnapshot = (0, firestore_1.query)(documentApprovalsRef, (0, firestore_1.and)((0, firestore_1.where)("workspaceId", "==", workspaceId), (0, firestore_1.where)("isManual", "==", false)));
        const approvalsSnapshot = yield (0, firestore_1.getDocs)(querySnapshot);
        if (approvalsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No document approvals found for this workspace.",
            });
        }
        const approvals = [];
        const usersRef = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        for (const approvalDoc of approvalsSnapshot.docs) {
            const data = approvalDoc.data();
            // Check if the current user is a receiver
            const matchingReceiver = data.receiver.find((receiver) => receiver.id === user.id);
            if (!matchingReceiver) {
                // Skip if the user is not a receiver
                continue;
            }
            // Fetch requester details
            const requesterSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, data.requesterId));
            const requesterData = requesterSnapshot.exists()
                ? requesterSnapshot.data()
                : null;
            // Fetch details for the matching receiver
            const receiverSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, matchingReceiver.id));
            const receiverData = receiverSnapshot.exists()
                ? {
                    id: matchingReceiver.id,
                    name: receiverSnapshot.data().name || null,
                    photoURL: receiverSnapshot.data().photoURL || null,
                    email: receiverSnapshot.data().email || null,
                    slug: matchingReceiver.slug,
                    isSigned: matchingReceiver.isSigned,
                }
                : null;
            approvals.push({
                documentData: data,
                requester: requesterData
                    ? {
                        id: data.requesterId,
                        name: requesterData.name || null,
                        photoURL: requesterData.photoURL || null,
                        email: requesterData.email || null,
                    }
                    : null,
                receiver: receiverData || null,
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            data: approvals,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.viewDocumentReceiverApprovals = viewDocumentReceiverApprovals;
const viewManualDocumentApprovals = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.id || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Unauthorized access.",
            });
        }
        const documentApprovalsRef = (0, firestore_1.collection)(firebase_1.db, "documentApprovals");
        const querySnapshot = (0, firestore_1.query)(documentApprovalsRef, (0, firestore_1.and)((0, firestore_1.where)("workspaceId", "==", workspaceId), (0, firestore_1.where)("isManual", "==", true)));
        const approvalsSnapshot = yield (0, firestore_1.getDocs)(querySnapshot);
        if (approvalsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No document approvals found for this workspace.",
            });
        }
        const approvals = [];
        const usersRef = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        for (const approvalDoc of approvalsSnapshot.docs) {
            const data = approvalDoc.data();
            // Check if the current user is the requester
            const isRequester = data.requesterId === user.id;
            if (!isRequester) {
                // Skip if the user is not the requester
                continue;
            }
            // Fetch requester details
            const requesterSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, data.requesterId));
            const requesterData = requesterSnapshot.exists()
                ? requesterSnapshot.data()
                : null;
            // Fetch details for all receivers
            const receivers = yield Promise.all(data.receiver.map((receiver) => __awaiter(void 0, void 0, void 0, function* () {
                const receiverSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, receiver.id));
                return receiverSnapshot.exists()
                    ? {
                        id: receiver.id,
                        name: receiverSnapshot.data().name || null,
                        photoURL: receiverSnapshot.data().photoURL || null,
                        email: receiverSnapshot.data().email || null,
                        slug: receiver.slug,
                        isSigned: receiver.isSigned,
                    }
                    : null;
            })));
            approvals.push({
                documentData: data,
                requester: requesterData
                    ? {
                        id: data.requesterId,
                        name: requesterData.name || null,
                        photoURL: requesterData.photoURL || null,
                        email: requesterData.email || null,
                    }
                    : null,
                receiver: receivers.filter((receiver) => receiver !== null),
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            data: approvals,
        });
    }
    catch (error) {
        console.error(error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.viewManualDocumentApprovals = viewManualDocumentApprovals;
const viewManualDocumentReceiverApprovals = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        if (!user || !user.id || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Unauthorized access.",
            });
        }
        const documentApprovalsRef = (0, firestore_1.collection)(firebase_1.db, "documentApprovals");
        const querySnapshot = (0, firestore_1.query)(documentApprovalsRef, (0, firestore_1.and)((0, firestore_1.where)("workspaceId", "==", workspaceId), (0, firestore_1.where)("isManual", "==", true)));
        const approvalsSnapshot = yield (0, firestore_1.getDocs)(querySnapshot);
        if (approvalsSnapshot.empty) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No document approvals found for this workspace.",
            });
        }
        const approvals = [];
        const usersRef = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        for (const approvalDoc of approvalsSnapshot.docs) {
            const data = approvalDoc.data();
            // Check if the current user is a receiver
            const matchingReceiver = data.receiver.find((receiver) => receiver.id === user.id);
            if (!matchingReceiver) {
                // Skip if the user is not a receiver
                continue;
            }
            // Fetch requester details
            const requesterSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, data.requesterId));
            const requesterData = requesterSnapshot.exists()
                ? requesterSnapshot.data()
                : null;
            // Fetch details for the matching receiver
            const receiverSnapshot = yield (0, firestore_1.getDoc)((0, firestore_1.doc)(usersRef, matchingReceiver.id));
            const receiverData = receiverSnapshot.exists()
                ? {
                    id: matchingReceiver.id,
                    name: receiverSnapshot.data().name || null,
                    photoURL: receiverSnapshot.data().photoURL || null,
                    email: receiverSnapshot.data().email || null,
                    slug: matchingReceiver.slug,
                    isSigned: matchingReceiver.isSigned,
                }
                : null;
            approvals.push({
                documentData: data,
                requester: requesterData
                    ? {
                        id: data.requesterId,
                        name: requesterData.name || null,
                        photoURL: requesterData.photoURL || null,
                        email: requesterData.email || null,
                    }
                    : null,
                receiver: receiverData || null,
            });
        }
        return res.status(200).json({
            status: "success",
            code: 200,
            data: approvals,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.viewManualDocumentReceiverApprovals = viewManualDocumentReceiverApprovals;
const sendManualDocumentApproval = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const { workspaceId, documentId, divisiFolderId } = req.params;
        const { userId } = req.body;
        const user = req.user;
        // Validate required parameters
        if (!workspaceId || !documentId || !divisiFolderId || !userId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID, Document ID, Folder ID, and Superior ID are required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to send document approvals.",
            });
        }
        // Fetch workspace and document
        const [workspaceDoc, documentDoc] = yield Promise.all([
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "workspaces", workspaceId)),
            (0, firestore_1.getDoc)((0, firestore_1.doc)(firebase_1.db, "documents", documentId)),
        ]);
        if (!workspaceDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found.",
            });
        }
        if (!documentDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document not found.",
            });
        }
        const documentData = documentDoc.data();
        if (documentData.divisiFolderId !== divisiFolderId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Document does not belong to the specified folder.",
            });
        }
        const filePath = documentData === null || documentData === void 0 ? void 0 : documentData.storagePath;
        const fileExtension = (documentData === null || documentData === void 0 ? void 0 : documentData.fileExtension) || "docx";
        const documentTitle = (documentData === null || documentData === void 0 ? void 0 : documentData.title) || "Untitled Document";
        if (!filePath) {
            return res.status(500).json({
                status: "error",
                message: "Document storage path is missing in the database.",
            });
        }
        // Check user permissions
        const { access } = documentData;
        const hasPermission = access.ownerId === user.id ||
            ((_a = access.allowedRoles) === null || _a === void 0 ? void 0 : _a.includes(user.role)) ||
            ((_b = access.invitedUsers) === null || _b === void 0 ? void 0 : _b.some((invitedUser) => invitedUser.id === user.id &&
                ["editor", "viewer"].includes(invitedUser.role)));
        if (!hasPermission) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You do not have permission to request approvals for this document.",
            });
        }
        // Prepare receiver data (superiors)
        const usersCollection = (0, firestore_1.collection)(firebase_1.db, `companies/${user.companyId}/users`);
        const superiorDocs = yield Promise.all((Array.isArray(userId) ? userId : [userId]).map((id) => (0, firestore_1.getDoc)((0, firestore_1.doc)(usersCollection, id))));
        const superiorData = superiorDocs
            .filter((doc) => doc.exists())
            .map((doc) => doc.data());
        const receiver = superiorData.map((superior) => ({
            id: superior.id,
            name: superior.name,
            email: superior.email,
        }));
        if (superiorData.length === 0) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "No valid superior users found.",
            });
        }
        let pdfFilePath = null;
        // If the file is already a PDF
        if (fileExtension === "pdf") {
            pdfFilePath = filePath;
            console.log("file already PDF");
        }
        else if (documentData.documentPDF &&
            documentData.documentPDF.documentBaseVersion ===
                documentData.currentVersion) {
            console.log("file same version");
            pdfFilePath = documentData.documentPDF.storagePath;
        }
        else {
            // Perform conversion using ONLYOFFICE
            const storageFile = firebase_admin_1.bucket.file(filePath);
            const [signedUrl] = yield storageFile.getSignedUrl({
                action: "read",
                expires: Date.now() + 1000 * 60 * 60 * 24 * 7,
            });
            const key = generateDocumentKey();
            const payload = Object.assign({ async: false, filetype: fileExtension, key, outputtype: "pdf", title: documentTitle, url: signedUrl }, (cellExtensions.includes(fileExtension) && {
                spreadsheetLayout: {
                    ignorePrintArea: true,
                    orientation: "landscape",
                    fitToWidth: 1,
                    fitToHeight: 1,
                    scale: 100,
                    headings: false,
                    gridLines: false,
                    pageSize: {
                        width: "210mm",
                        height: "148mm",
                    },
                },
            }));
            const token = jsonwebtoken_1.default.sign(payload, process.env.JWT_SECRET_KEY_ONLYOFFICE || "");
            const conversionUrl = `${process.env.ONLYOFFICE_DOCUMENT_SERVER_URL}?shardkey=${key}`;
            const conversionResponse = yield axios_1.default.post(conversionUrl, Object.assign(Object.assign({}, payload), { token }));
            if (conversionResponse.data.error ||
                !conversionResponse.data.endConvert) {
                return res.status(500).json({
                    status: "error",
                    message: `Conversion failed: ${conversionResponse.data.error}`,
                });
            }
            const pdfResponse = yield fetch(conversionResponse.data.fileUrl);
            if (!pdfResponse.ok) {
                return res.status(500).json({
                    status: "error",
                    message: "Failed to download converted PDF from ONLYOFFICE.",
                });
            }
            const pdfBuffer = yield pdfResponse.arrayBuffer();
            pdfFilePath = `${filePath.replace(/\/[^/]+$/, "")}/pdf/${documentTitle}.pdf`;
            yield firebase_admin_1.bucket.file(pdfFilePath).save(Buffer.from(pdfBuffer), {
                resumable: false,
                contentType: "application/pdf",
            });
            // Update document with the new PDF info
            yield (0, firestore_1.updateDoc)((0, firestore_1.doc)(firebase_1.db, "documents", documentId), {
                documentPDF: {
                    documentBaseVersion: documentData.currentVersion,
                    storagePath: pdfFilePath,
                    createdAt: new Date().toISOString(),
                },
            });
        }
        // Create entry in `documentApproval` collection
        const documentApprovalRef = firebase_admin_1.adminDb.collection("documentApprovals").doc();
        yield documentApprovalRef.set({
            id: documentApprovalRef.id,
            title: documentTitle,
            access: {
                ownerId: user.id,
                ownerName: user.name,
                allowedRoles: ["superadmin", "admin"],
                invitedUsers: receiver,
            },
            workspaceId,
            documentId,
            isManual: true,
            key: generateDocumentKey(),
            currentVersion: 1,
            divisiFolderId,
            requesterId: user.id,
            receiver,
            status: "Dalam Proses",
            requestedAt: new Date().toISOString(),
            fileExtension: "pdf",
            storagePath: pdfFilePath,
        });
        const notificationRecipients = superiorData.map((superior) => ({
            id: superior.id,
            name: superior.name,
            email: superior.email,
            photoURL: (superior === null || superior === void 0 ? void 0 : superior.photoURL) || undefined,
            isRead: false,
        }));
        // Notification message
        const notificationMessage = `You have been requested to review and approve the document titled "${documentTitle}".`;
        // Send notifications to the `receiver`
        yield (0, notifications_1.sendNotificationDocumentApproval)(documentApprovalRef.id, documentId, documentTitle, user.companyId, user.id, user.name, user.photoURL || undefined, notificationRecipients, workspaceId, true, "document approval submitted", notificationMessage, "approvalNotification");
        const emailLink = `${process.env.APP_URL}/document-approval`;
        yield Promise.all([
            ...superiorData.map((superior) => (0, email_1.sendEmailApproval)({
                toEmail: superior.email,
                fromName: user.name,
                recipientName: superior.name,
                link: `${emailLink}/manual-signs`,
                message: `You have been requested to review and approve the document titled "${documentTitle}". \nThank you.`,
            })),
            (0, email_1.sendEmailApproval)({
                toEmail: user.email,
                fromName: "System Notification",
                recipientName: user.name,
                link: `${emailLink}/manual-requests`,
                message: `Your request for document approval has been successfully sent for the document titled "${documentTitle}".\nThank you.`,
            }),
        ]);
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval request sent successfully.",
            data: {
                storagePath: pdfFilePath,
                receiver,
            },
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.sendManualDocumentApproval = sendManualDocumentApproval;
const editDocumentApproval = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { user } = req;
        const { approvalId } = req.params;
        if (!approvalId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Approval ID is required.",
            });
        }
        const approvalDocRef = firebase_admin_1.adminDb
            .collection("documentApprovals")
            .doc(approvalId);
        const approvalDoc = yield approvalDocRef.get();
        if (!approvalDoc.exists) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document approval not found.",
            });
        }
        const approvalData = approvalDoc.data();
        // Find the receiver corresponding to the user.id
        const receiverIndex = approvalData.receiver.findIndex((receiver) => receiver.id === user.id);
        if (receiverIndex === -1) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You are not authorized to sign this document.",
            });
        }
        const receiver = approvalData.receiver[receiverIndex];
        if (receiver.isSigned) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "You have already signed this document.",
            });
        }
        // Update the specific receiver to mark as signed
        const updatedReceivers = [...approvalData.receiver];
        updatedReceivers[receiverIndex] = Object.assign(Object.assign({}, receiver), { isSigned: true });
        // Check if all or any receivers have signed
        const allSigned = updatedReceivers.every((r) => r.isSigned);
        const anySigned = updatedReceivers.some((r) => r.isSigned);
        // Determine the status
        let status = approvalData.status;
        if (allSigned) {
            status = "disetujui";
        }
        else if (anySigned) {
            status = "pending";
        }
        // Update the document
        yield approvalDocRef.update({
            receiver: updatedReceivers,
            status,
            updatedAt: new Date().toISOString(),
        });
        let message = "";
        if (status === "disetujui") {
            message = `The document "${approvalData.title}" has been signed by all participants.`;
        }
        else {
            message = `The document "${approvalData.title}" has been signed by ${receiver.name}`;
        }
        const recipients = updatedReceivers.map((receiver) => ({
            id: receiver.id,
            name: receiver.name,
            email: receiver.email,
            photoURL: receiver.photoURL || null,
            isRead: false
        }));
        yield (0, notifications_1.sendNotificationDocumentApproval)(approvalId, // Document Approval ID
        approvalData.documentId, // Document ID
        approvalData.title, user.companyId, user.id, user.name, (user === null || user === void 0 ? void 0 : user.photoURL) || undefined, recipients, approvalData.workspaceId, true, "document approval signed", message, "approvalNotification");
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval updated successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.editDocumentApproval = editDocumentApproval;
const deleteDocumentApproval = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { documentApprovalId } = req.params;
        const user = req.user;
        if (!documentApprovalId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Document approval ID is required.",
            });
        }
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to delete document approvals.",
            });
        }
        // Fetch the document approval
        const documentApprovalRef = (0, firestore_1.doc)(firebase_1.db, "documentApprovals", documentApprovalId);
        const documentApprovalDoc = yield (0, firestore_1.getDoc)(documentApprovalRef);
        if (!documentApprovalDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Document approval not found.",
            });
        }
        const documentApprovalData = documentApprovalDoc.data();
        const { documentId, requesterId } = documentApprovalData;
        if (requesterId !== user.id) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "You are not authorized to delete this document approval. Only the requester can perform this action.",
            });
        }
        const documentRef = (0, firestore_1.doc)(firebase_1.db, "documents", documentId);
        const documentDoc = yield (0, firestore_1.getDoc)(documentRef);
        if (!documentDoc.exists()) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Associated document not found.",
            });
        }
        const documentData = documentDoc.data();
        const { fileExtension, documentPDF } = documentData;
        if (fileExtension !== "pdf" && (documentPDF === null || documentPDF === void 0 ? void 0 : documentPDF.storagePath)) {
            const storageFile = firebase_admin_1.bucket.file(documentPDF.storagePath);
            try {
                yield storageFile.delete();
                console.log(`File deleted successfully: ${documentPDF.storagePath}`);
            }
            catch (error) {
                console.error("Error deleting file from Storage:", error);
            }
            // Remove the documentPDF field from the document
            yield (0, firestore_1.updateDoc)(documentRef, { documentPDF: null });
        }
        // Delete the document approval record
        yield (0, firestore_1.deleteDoc)(documentApprovalRef);
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Document approval and related data have been deleted successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error.",
        });
    }
});
exports.deleteDocumentApproval = deleteDocumentApproval;
