"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFoldersAndCategoryDocumentCounts = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const getFoldersAndCategoryDocumentCounts = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const { workspaceId } = req.params;
        const user = req.user;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        // Verify the workspace
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists ||
            ((_a = workspaceDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Workspace not found or access denied.",
            });
        }
        // Fetch all folders
        const foldersSnapshot = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .collection("folders")
            .get();
        // Initialize folder data and total document count
        let totalDocuments = 0;
        const folderData = yield Promise.all(foldersSnapshot.docs.map((folderDoc) => __awaiter(void 0, void 0, void 0, function* () {
            const folderData = folderDoc.data();
            const documentSnapshot = yield firebase_admin_1.adminDb
                .collection("documents")
                .where("divisiFolderId", "==", folderDoc.id)
                .get();
            const documentCount = documentSnapshot.size;
            totalDocuments += documentCount;
            return {
                folderId: folderDoc.id,
                folderName: folderData.folderName,
                documentCount,
            };
        })));
        // Fetch all documents in the workspace for category counting
        const documentsSnapshot = yield firebase_admin_1.adminDb
            .collection("documents")
            .where("workspaceId", "==", workspaceId)
            .get();
        // Initialize category count map
        const categoryCountMap = {};
        // Document status counters
        let approvedCount = 0;
        let rejectedCount = 0;
        let inProgressCount = 0;
        // Count documents per category and status
        documentsSnapshot.forEach((doc) => {
            var _a;
            const data = doc.data();
            const categoryId = data.categoryId || "Uncategorized";
            categoryCountMap[categoryId] = (categoryCountMap[categoryId] || 0) + 1;
            const statusLower = (_a = data.status) === null || _a === void 0 ? void 0 : _a.toLowerCase();
            if (statusLower === "disetujui") {
                approvedCount++;
            }
            else if (statusLower === "ditolak") {
                rejectedCount++;
            }
            else if (statusLower === "dalam proses") {
                inProgressCount++;
            }
        });
        // Fetch category names
        const categoryIds = Object.keys(categoryCountMap);
        const categorySnapshots = yield Promise.all(categoryIds.map((id) => firebase_admin_1.adminDb.collection("categories").doc(id).get()));
        const categoryData = categorySnapshots.map((snap) => {
            var _a;
            const categoryName = snap.exists
                ? (_a = snap.data()) === null || _a === void 0 ? void 0 : _a.categoryName
                : "Uncategorized";
            const categoryId = snap.id;
            const documentCount = categoryCountMap[categoryId] || 0;
            return {
                categoryId,
                categoryName,
                documentCount,
                percentage: totalDocuments ? (documentCount / totalDocuments) * 100 : 0,
            };
        });
        // Calculate percentage for each folder
        const folderDataWithPercentage = folderData.map((folder) => (Object.assign(Object.assign({}, folder), { percentage: totalDocuments
                ? (folder.documentCount / totalDocuments) * 100
                : 0 })));
        // Prepare status counts as an array
        const statusCounts = [
            {
                label: "approved",
                documentCount: approvedCount,
                percentage: totalDocuments ? (approvedCount / totalDocuments) * 100 : 0,
            },
            {
                label: "rejected",
                documentCount: rejectedCount,
                percentage: totalDocuments ? (rejectedCount / totalDocuments) * 100 : 0,
            },
            {
                label: "inProgress",
                documentCount: inProgressCount,
                percentage: totalDocuments
                    ? (inProgressCount / totalDocuments) * 100
                    : 0,
            },
        ];
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Folders, categories, and document statuses fetched successfully.",
            data: {
                folders: folderDataWithPercentage,
                categories: categoryData,
                totalDocuments,
                statusCounts,
            },
        });
    }
    catch (error) {
        console.error("Error fetching folder and document counts:", error);
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.getFoldersAndCategoryDocumentCounts = getFoldersAndCategoryDocumentCounts;
