"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCategory = exports.viewAllCategories = exports.editCategories = exports.createCategory = void 0;
const firebase_admin_1 = require("../../../../utils/firebase-admin");
const zod_1 = require("zod");
const category_1 = require("../../../../types/category");
// Helper function to validate user and companyId
const validateUserAndCompany = (user, res) => {
    if (!user || !user.companyId) {
        return res.status(403).json({
            status: "error",
            code: 403,
            message: "User is not authorized to perform this action.",
        });
    }
};
// Create a new category
const createCategory = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    try {
        const user = req.user;
        validateUserAndCompany(user, res);
        const parsedBody = category_1.createCategorySchema.parse(req.body);
        const { workspaceId } = req.params;
        const { categoryName } = parsedBody;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        // Check if the workspace belongs to the same company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists ||
            ((_a = workspaceDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Workspace not found or access denied.",
            });
        }
        // Check if a category with the same name exists within the same workspace
        const existingCategorySnapshot = yield firebase_admin_1.adminDb
            .collection("categories")
            .where("categoryName", "==", categoryName)
            .where("workspaceId", "==", workspaceId)
            .limit(1)
            .get();
        if (!existingCategorySnapshot.empty) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "A category with the same name already exists in this workspace.",
            });
        }
        // Create new category
        const categoryRef = firebase_admin_1.adminDb.collection("categories").doc();
        yield categoryRef.set({
            id: categoryRef.id,
            categoryName,
            workspaceId,
            companyId: user.companyId,
            createdAt: new Date().toISOString(),
            createdBy: user.id,
            updatedByName: user.name,
            updatedByEmail: user.email,
        });
        return res.status(201).json({
            status: "success",
            code: 201,
            message: "Category created successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.createCategory = createCategory;
const editCategories = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const user = req.user;
        validateUserAndCompany(user, res);
        const { categoryId, workspaceId } = req.params;
        if (!categoryId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Category ID is required.",
            });
        }
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        const parsedBody = category_1.editCategorySchema.parse(req.body);
        const { categoryName } = parsedBody;
        // Check if the workspace belongs to the same company
        const workspaceDoc = yield firebase_admin_1.adminDb
            .collection("workspaces")
            .doc(workspaceId)
            .get();
        if (!workspaceDoc.exists ||
            ((_a = workspaceDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "Workspace not found or access denied.",
            });
        }
        // Check if the category exists and belongs to the user's company
        const categoryDoc = yield firebase_admin_1.adminDb
            .collection("categories")
            .doc(categoryId)
            .get();
        if (!categoryDoc.exists ||
            ((_b = categoryDoc.data()) === null || _b === void 0 ? void 0 : _b.companyId) !== user.companyId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Category not found or access denied.",
            });
        }
        // Update the category and include updatedByName and updatedByEmail
        yield categoryDoc.ref.update(Object.assign(Object.assign({}, (categoryName && { categoryName })), { updatedAt: new Date().toISOString(), updatedBy: user.id, updatedByName: user.name, updatedByEmail: user.email }));
        // Fetch and update all documents with the specified categoryId
        const documentsSnapshot = yield firebase_admin_1.adminDb
            .collection("documents")
            .where("categoryId", "==", categoryId)
            .get();
        const updatePromises = documentsSnapshot.docs.map((doc) => doc.ref.update({ categoryName }));
        yield Promise.all(updatePromises); // Update all matching documents in parallel
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Category and related documents updated successfully.",
        });
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const firstError = error.errors[0];
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Validation error",
                error: firstError,
            });
        }
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.editCategories = editCategories;
const viewAllCategories = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const user = req.user;
        if (!user || !user.companyId) {
            return res.status(403).json({
                status: "error",
                code: 403,
                message: "User is not authorized to view categories.",
            });
        }
        const { workspaceId } = req.params;
        if (!workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Workspace ID is required.",
            });
        }
        const categoriesSnapshot = yield firebase_admin_1.adminDb
            .collection("categories")
            .where("companyId", "==", user.companyId)
            .where("workspaceId", "==", workspaceId)
            .get();
        if (categoriesSnapshot.empty) {
            return res.status(200).json({
                status: "success",
                code: 200,
                message: "No categories found.",
                data: [],
            });
        }
        // Fetch categories and createdBy details
        const categories = yield Promise.all(categoriesSnapshot.docs.map((doc) => __awaiter(void 0, void 0, void 0, function* () {
            const data = doc.data();
            const createdById = data.createdBy; // Ensure it's a string (user ID)
            // Fetch user who created the category
            let createdByUser = null;
            if (createdById) {
                const createdByUserDoc = yield firebase_admin_1.adminDb
                    .collection("companies")
                    .doc(user === null || user === void 0 ? void 0 : user.companyId)
                    .collection("users")
                    .doc(createdById) // Pass the user ID to the `doc` method
                    .get();
                if (createdByUserDoc.exists) {
                    const userData = createdByUserDoc.data();
                    createdByUser = {
                        id: createdById,
                        name: (userData === null || userData === void 0 ? void 0 : userData.name) || "Unknown User",
                        email: (userData === null || userData === void 0 ? void 0 : userData.email) || "No Email",
                        photoURL: (userData === null || userData === void 0 ? void 0 : userData.photoURL) || null,
                    };
                }
            }
            return Object.assign(Object.assign({ id: doc.id }, data), { createdByUser });
        })));
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Categories fetched successfully.",
            data: categories,
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.viewAllCategories = viewAllCategories;
const deleteCategory = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        const user = req.user;
        validateUserAndCompany(user, res);
        const { categoryId, workspaceId } = req.params;
        if (!categoryId || !workspaceId) {
            return res.status(400).json({
                status: "error",
                code: 400,
                message: "Category ID and Workspace ID are required.",
            });
        }
        // Check if the category exists and belongs to the user's company and workspace
        const categoryDoc = yield firebase_admin_1.adminDb
            .collection("categories")
            .doc(categoryId)
            .get();
        if (!categoryDoc.exists ||
            ((_a = categoryDoc.data()) === null || _a === void 0 ? void 0 : _a.companyId) !== user.companyId ||
            ((_b = categoryDoc.data()) === null || _b === void 0 ? void 0 : _b.workspaceId) !== workspaceId) {
            return res.status(404).json({
                status: "error",
                code: 404,
                message: "Category not found or access denied.",
            });
        }
        // Delete the category
        yield categoryDoc.ref.delete();
        return res.status(200).json({
            status: "success",
            code: 200,
            message: "Category deleted successfully.",
        });
    }
    catch (error) {
        return res.status(500).json({
            status: "error",
            code: 500,
            message: "Internal Server Error",
        });
    }
});
exports.deleteCategory = deleteCategory;
